"""
MoinLinks Extension
===================

Adds MoinMoin-styled link syntax.
e.g. [[WikiLink]] or [[WikiLink | Description]] or [[attachment:foo.txt]]

Based on wikilinks extension in Python-Markdown by [Waylan Limberg](http://achinghead.com/),
modified to pass whole link pattern to default wiki parser from MoinMoin.
Everything that MoinMoin parser does are allowed, such as:

- various styles for different states of the link target (nonexistance, interwiki, etc)
- link descriptions
- attachment linking

We need running MoinMoin environment to use this extension.
See text_markdown plugin from the same author for details on how to get
this to work.

Copyright 2014 dossist.

"""

from __future__ import absolute_import
from __future__ import unicode_literals
from markdown.extensions import Extension
from markdown.inlinepatterns import Pattern
from markdown.util import etree
from markdown.util import AtomicString

from MoinMoin.parser.text_moin_wiki import Parser as TextMoinWikiParser
import StringIO

class MoinLinksExtension(Extension):
    # WikiLink pattern to match
    MOINLINK_RE = r'(\[\[[^\n]+?\]\])'
    moinlinksLabel = 'moinlink'
    moinlinksPosition = "<not_strong"

    def __init__(self, *args, **kwargs):
        # default config values
        self.config = {
            'request': ['', 'original request from MoinMoin'],
        }
        super(MoinLinksExtension, self).__init__(*args, **kwargs)

    def extendMarkdown(self, md, md_globals):
        self.md = md
        # append pattern to end of inline patterns
        moinlinksPattern = MoinLinks(self.MOINLINK_RE, self.getConfigs())
        moinlinksPattern.md = md
        md.inlinePatterns.add(self.moinlinksLabel, moinlinksPattern, self.moinlinksPosition)


class MoinLinks(Pattern):
    def __init__(self, pattern, config):
        super(MoinLinks, self).__init__(pattern)
        self.config = config

    def handleMatch(self, m):
        """ let moin wiki parser convert the moin link into html and reparse it into ElementTree """

        request = self.config['request']

        tempstream = StringIO.StringIO()
        request.redirect(tempstream)
        # matching pattern has been modified to have one extra maching group before our own pattern
        parser = TextMoinWikiParser(m.group(2), request, line_anchors=False)
        parser.format(request.formatter, inhibit_p=True)
        # sandwich output between <span>~</span> to avoid error when re-parsing
        moinout = u''.join((u'<span>', tempstream.getvalue(), u'</span>')) #unicode
        request.redirect()

        tree = etree.fromstring(moinout.encode("utf-8"))
        # first child of tree is what we need
        if len(tree):
            out = tree[0] # first child
        else:
            out = tree # parent span element: THIS SHOULD NOT HAPPEN!
        if out.text:
            out.text = AtomicString(out.text)

        return out


def makeExtension(*args, **kwargs):
    return MoinLinksExtension(*args, **kwargs)
